<?php
/**
 * API endpoint for image management - Sử dụng config.php
 */

// Load cấu hình chung
require_once __DIR__ . '/config.php';

$method = $_SERVER['REQUEST_METHOD'];
$uploadDir = '../public/images/news/';

try {
    switch ($method) {
        case 'GET':
            handleGetImages($uploadDir);
            break;
        case 'DELETE':
            handleDeleteImage($uploadDir);
            break;
        default:
            returnJson(['success' => false, 'message' => 'Method not allowed'], 405);
    }
} catch (Exception $e) {
    returnJson([
        'success' => false,
        'message' => 'Server error: ' . $e->getMessage()
    ], 500);
}

function handleGetImages($uploadDir) {
    // Tạo thư mục nếu chưa có
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0755, true);
        returnJson([
            'success' => true,
            'images' => [],
            'total' => 0,
            'message' => 'Upload directory created'
        ]);
        return;
    }
    
    $images = [];
    $files = scandir($uploadDir);
    
    // Generate base URL
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $baseUrl = $protocol . '://' . $host;
    
    foreach ($files as $file) {
        if ($file === '.' || $file === '..' || $file === '.gitkeep') {
            continue;
        }
        
        $filePath = $uploadDir . $file;
        if (is_file($filePath)) {
            // Check if it's an image file
            $imageExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'jfif'];
            $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
            
            if (in_array($extension, $imageExtensions)) {
                $fileSize = filesize($filePath);
                $fileTime = filemtime($filePath);
                
                // Try to get image dimensions
                $dimensions = null;
                $imageInfo = @getimagesize($filePath);
                if ($imageInfo !== false) {
                    $dimensions = [
                        'width' => $imageInfo[0],
                        'height' => $imageInfo[1]
                    ];
                }
                
                $images[] = [
                    'filename' => $file,
                    'url' => $baseUrl . '/public/images/news/' . $file,
                    'size' => $fileSize,
                    'size_formatted' => formatFileSize($fileSize),
                    'created_at' => date('Y-m-d H:i:s', $fileTime),
                    'extension' => $extension,
                    'dimensions' => $dimensions
                ];
            }
        }
    }
    
    // Sort by creation time (newest first)
    usort($images, function($a, $b) {
        return strtotime($b['created_at']) - strtotime($a['created_at']);
    });
    
    returnJson([
        'success' => true,
        'images' => $images,
        'total' => count($images),
        'total_size' => array_sum(array_column($images, 'size')),
        'total_size_formatted' => formatFileSize(array_sum(array_column($images, 'size')))
    ]);
}

function handleDeleteImage($uploadDir) {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($input['filename']) || empty($input['filename'])) {
        returnJson(['success' => false, 'message' => 'Filename is required'], 400);
        return;
    }
    
    $filename = $input['filename'];
    
    // Validate filename (prevent directory traversal)
    if (strpos($filename, '..') !== false || strpos($filename, '/') !== false || strpos($filename, '\\') !== false) {
        returnJson(['success' => false, 'message' => 'Invalid filename'], 400);
        return;
    }
    
    $filePath = $uploadDir . $filename;
    
    if (!file_exists($filePath)) {
        returnJson(['success' => false, 'message' => 'File not found'], 404);
        return;
    }
    
    if (!is_file($filePath)) {
        returnJson(['success' => false, 'message' => 'Not a file'], 400);
        return;
    }
    
    // Delete the file
    if (unlink($filePath)) {
        returnJson([
            'success' => true,
            'message' => 'Image deleted successfully',
            'filename' => $filename
        ]);
    } else {
        returnJson(['success' => false, 'message' => 'Failed to delete file'], 500);
    }
}

function formatFileSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' bytes';
    }
}
?>
