<?php
/**
 * API endpoint for image upload - Sử dụng config.php
 */

// Load cấu hình chung
require_once __DIR__ . '/config.php';

// Cấu hình upload
$uploadConfig = [
    'image' => [
        'max_size' => 20 * 1024 * 1024, // 20MB
        'max_size_text' => '20MB'
    ]
];

// Xử lý upload file thông thường
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['image'])) {
    handleFileUpload($_FILES['image']);
    exit();
}

// Xử lý upload base64
$input = json_decode(file_get_contents('php://input'), true);

// Check action
if (!isset($input['action']) || $input['action'] !== 'upload_image') {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid action']);
    exit();
}

// Validate required fields
$requiredFields = ['image_data', 'filename', 'type'];
foreach ($requiredFields as $field) {
    if (!isset($input[$field]) || empty($input[$field])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => "Missing required field: {$field}"]);
        exit();
    }
}

try {
    // Decode base64 image data
    $imageData = base64_decode($input['image_data']);
    if ($imageData === false) {
        throw new Exception('Invalid image data');
    }
    
    // Validate file size using config
    $maxSize = $uploadConfig['image']['max_size'];
    $maxSizeText = $uploadConfig['image']['max_size_text'];
    
    if (strlen($imageData) > $maxSize) {
        throw new Exception("File quá lớn. Kích thước tối đa: {$maxSizeText}");
    }
    
    // Validate image data
    $imageInfo = getimagesizefromstring($imageData);
    if ($imageInfo === false) {
        throw new Exception('Invalid image format');
    }
    
    // Create upload directory
    $uploadDir = '../public/images/news/';
    if (!is_dir($uploadDir)) {
        if (!mkdir($uploadDir, 0755, true)) {
            throw new Exception('Cannot create upload directory');
        }
    }
    
    // Save image file
    $filename = $input['filename'];
    $filePath = $uploadDir . $filename;
    
    if (file_put_contents($filePath, $imageData) === false) {
        throw new Exception('Cannot save image file');
    }
    
    // Generate URL for the uploaded file
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $baseUrl = $protocol . '://' . $host;
    $fileUrl = $baseUrl . '/public/images/news/' . $filename;
    
    // Return success response
    echo json_encode([
        'success' => true,
        'message' => 'Image uploaded successfully',
        'url' => $fileUrl,
        'filename' => $filename,
        'size' => strlen($imageData),
        'dimensions' => [
            'width' => $imageInfo[0],
            'height' => $imageInfo[1]
        ]
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Upload failed: ' . $e->getMessage()
    ]);
}

// Function xử lý upload file thông thường
function handleFileUpload($file) {
    global $uploadConfig;
    
    try {
        // Validate file
        if ($file['error'] !== UPLOAD_ERR_OK) {
            throw new Exception('Upload error: ' . $file['error']);
        }
        
        // Validate file size
        if ($file['size'] > $uploadConfig['image']['max_size']) {
            throw new Exception('File quá lớn. Kích thước tối đa: ' . $uploadConfig['image']['max_size_text']);
        }
        
        // Validate file type
        $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        if (!in_array($file['type'], $allowedTypes)) {
            throw new Exception('Định dạng file không được hỗ trợ');
        }
        
        // Create upload directory
        $uploadDir = '../public/images/news/';
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        
        // Generate unique filename
        $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
        $filename = uniqid() . '.' . $extension;
        $filePath = $uploadDir . $filename;
        
        // Move uploaded file
        if (move_uploaded_file($file['tmp_name'], $filePath)) {
            // Get image info
            $imageInfo = getimagesize($filePath);
            
            // Generate URL
            $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
            $host = $_SERVER['HTTP_HOST'];
            $baseUrl = $protocol . '://' . $host;
            $fileUrl = $baseUrl . '/public/images/news/' . $filename;
            
            returnJson([
                'success' => true,
                'message' => 'Upload thành công',
                'url' => $fileUrl,
                'filename' => $filename,
                'size' => $file['size'],
                'dimensions' => [
                    'width' => $imageInfo[0],
                    'height' => $imageInfo[1]
                ]
            ]);
        } else {
            throw new Exception('Không thể lưu file');
        }
    } catch (Exception $e) {
        returnJson([
            'success' => false,
            'message' => 'Upload thất bại: ' . $e->getMessage()
        ], 500);
    }
}
?>
